import logging
from singleton import Singleton
from database_manager import database_manager
from mqtt_manager import mqtt_manager
import sqlite3
import time
import platform
import threading
import copy
import math
from kivy.clock import Clock
from subprocess import call
import serial
import serial.threaded 
if platform.machine().startswith('arm'):
    import RPi.GPIO as GPIO

class BoxFramedPacket(serial.threaded.FramedPacket):
    def __init__(self):
        super(BoxFramedPacket, self).__init__()
        self.START = '\x02'
        self.STOP = '\x03'
        self.received_packets = []
        self.lock = threading.Lock()

    def handle_packet(self, packet):
        try:
            msg = packet[:-2]
            chk = packet[-2:]
            if chk.upper() == self.checksum(msg):
                with self.lock:
                    self.received_packets.append(msg)
        except:
            pass

    def checksum(self, packet):
        checksum = 0
        for el in packet:
            checksum ^= ord(el)
        return str(bytearray([checksum])).encode('hex').upper()

    def send_packet(self, packet):
        with self.lock:
            self.received_packets = []
        self.transport.write(self.START)
        self.transport.write(packet + self.checksum(packet))
        self.transport.write(self.STOP)

    def get_received_packets(self):
        with self.lock:
            packets = copy.deepcopy(self.received_packets)
            self.received_packets = []
            return packets

class device_manager(Singleton):
    output_channels = [23,24]
    logger = None

    def __init__(self):
        self.logger = logging.getLogger('root')
        self.db = database_manager()
        self.boxes_selected = False
        self.card_reader_selected = False
        self.poll_addr = 0
        self.temp_targets = {}
        self.ser = None
        self.reader_thread = None
        self.protocol = None
        self.send_status_event = None
        if platform.machine().startswith('arm'):
            try:
                GPIO.setmode(GPIO.BCM)
                GPIO.setwarnings(False)
                GPIO.setup(self.output_channels, GPIO.OUT)
                GPIO.output(self.output_channels, GPIO.HIGH)
                self.logger.info('Revisione scheda: ' + GPIO.RPI_INFO['TYPE'])
            except:
                self.logger.error("Impossibile attivare GPIO")

            try:
                self.ser = serial.Serial('/dev/serial0', 9600, timeout=1)
                self.reader_thread = serial.threaded.ReaderThread(self.ser, BoxFramedPacket)
                self.protocol = self.reader_thread.__enter__()
                self.poll_event = Clock.schedule_once(self.polling_callback, 1)
            except:
                self.ser = None
                self.reader_thread = None
                self.protocol = None
                self.logger.error("Impossibile aprire porta seriale")
                time.sleep(60)
                # Scommentare in produzione
                #if platform.machine().startswith('arm'):
                #    call(["sudo", "reboot"])

    def __del__(self):
        if self.reader_thread is not None:
            self.reader_thread.__exit__(None, None, None)

    def select_boxes(self):
        self.parse_polling_packets()
        self.card_reader_selected = False
        try:
            GPIO.output(24, GPIO.HIGH)
            GPIO.output(23, GPIO.LOW)
            self.boxes_selected = True
        except:
            self.boxes_selected = False
            self.logger.error("Impossibile attivare GPIO")

    def select_card_reader(self):
        self.parse_polling_packets()
        self.boxes_selected = False
        try:
            GPIO.output(23, GPIO.HIGH)
            GPIO.output(24, GPIO.LOW)
            self.card_reader_selected = True
        except:
            self.card_reader_selected = False
            self.logger.error("Impossibile attivare GPIO")

    def log_available_boxes(self, boxes_db):
        self.logger.debug("Box disponibili:")
        for addr in boxes_db.keys():
            self.logger.debug(boxes_db[addr])

    def connect(self):
        boxes_db = self.get_boxes_db()
        # Se non ha ancora mai fatto un autoriconoscimento dei box, lo fa adesso
        if len(boxes_db) == 0:
            self.detect_boxes()
            boxes_db = self.get_boxes_db()
        else:
            mqtt = mqtt_manager()
            mqtt.send_status()
        self.log_available_boxes(boxes_db)
        self.send_status_event = Clock.schedule_interval(self.send_status_callback, 60)

    def write(self, msg):
        try:
            self.ser.write("\x02{0}{1}\x03".format(msg, self.checksum(msg)))
            x = self.ser.read(1024)
            return x
        except:
            self.logger.error("Errore porta seriale")
            return None

    def parse_polling_packets(self):
        is_db_updated = False
        for packet in self.protocol.get_received_packets():
            if len(packet) >= len("C0160024002") and packet[0:1] == "C" and packet[3:4] == "6":
                addr = int(packet[1:3])
                temperature = int(packet[4:8])
                c = self.db.get_conn().cursor()
                c.execute("update boxes set temperature = {} where addr = {}".format(temperature, addr))
                c.close()
                is_db_updated = True
                # Controlla se il box e' in fault di temperatura non raggiunta
                if addr in self.temp_targets.keys():
                    boxes_db = self.get_boxes_db(addr)
                    if addr in boxes_db.keys():
                        now = time.time()
                        if now > self.temp_targets['time'] + 60 and math.fabs(self.temp_targets['temp'] - temperature) > 3:
                            c = self.db.get_conn().cursor()
                            c.execute("update boxes set temp_fault = 1 where addr = {} and temp_fault = 0".format(addr))
                            c.close()
        if is_db_updated:
            mqtt = mqtt_manager()
            mqtt.send_status()

    def get_boxes_db(self, addr = None):
        return self.db.get_boxes_db(addr)

    def detect_boxes(self):
        if self.protocol is not None:
            self.select_boxes()
            x = self.protocol.send_packet("A000")
            time.sleep(10)
            c = self.db.get_conn().cursor()
            c.execute("DELETE FROM boxes")
            c.close()
            for packet in self.protocol.get_received_packets():
                if len(packet) >= len("C0101331P105000") and packet[0:1] == "C" and packet[3:4] == "0":
                    addr = int(packet[1:3])
                    version = int(packet[4:7])
                    doors = int(packet[7:8])
                    type_ref = packet[8:9]
                    min_temp = int(packet[9:11])
                    max_temp = int(packet[11:13])
                    uv_present = int(packet[13:14])
                    led_present = int(packet[14:15])
                    c = self.db.get_conn().cursor()
                    c.execute("INSERT INTO boxes(addr, version, doors, type_ref, min_temp, max_temp, uv_present, led_present) VALUES({}, {}, {}, '{}', {}, {}, {}, {})".format(addr, version, doors, type_ref, min_temp, max_temp, uv_present, led_present))
                    c.close()
            mqtt = mqtt_manager()
            mqtt.send_status()

    def error_not_responding(self, addr):
        self.logger.error("Il box {} non risponde".format(addr))

    def open_door(self, addr, door1, door2 = False, door3 = False, door4 = False):
        if self.protocol is not None:
            self.select_boxes()
            x = self.protocol.send_packet("A{0:02}4{1}{2}{3}{4}".format(addr, '1' if door1 else '0', '1' if door2 else '0', '1' if door3 else '0', '1' if door4 else '0'))
            for i in range(0,50):
                for packet in self.protocol.get_received_packets():
                    self.logger.debug(packet)
                    return
                time.sleep(0.1)
            self.error_not_responding(addr)

    def enable(self, addr, enable):
        if self.protocol is not None:
            self.select_boxes()
            x = self.protocol.send_packet("A{0:02}1{1}".format(addr, '1' if enable else '0'))
            for i in range(0,50):
                for packet in self.protocol.get_received_packets():
                    self.logger.debug(packet)
                    return
                time.sleep(0.1)
            self.error_not_responding(addr)

    def get_status(self, addr):
        if self.protocol is not None:
            self.select_boxes()
            x = self.protocol.send_packet("A{0:02}6".format(addr))
            for i in range(0,50):
                for packet in self.protocol.get_received_packets():
                    self.logger.debug(packet)
                    return
                time.sleep(0.1)
            self.error_not_responding(addr)

    def set_temp(self, addr, temp):
        if self.protocol is not None:
            self.select_boxes()
            self.temp_targets[addr] = {'temp':temp, 'time':time.time()}
            x = self.protocol.send_packet("A{0:02}2{0:02}".format(addr, temp))
            for i in range(0,50):
                for packet in self.protocol.get_received_packets():
                    self.logger.debug(packet)
                    return
                time.sleep(0.1)
            self.error_not_responding(addr)

    def polling_callback(self, dt):
        if self.protocol is not None:
            boxes_db = self.get_boxes_db()
            # Calcolo dell'indirizzo di polling successivo
            new_poll_addr = 0
            for addr in boxes_db.keys():
                if addr > self.poll_addr:
                    new_poll_addr = addr
                    break
            if new_poll_addr == 0:
                # Riparte dal primo indirizzo
                for addr in boxes_db.keys():
                    new_poll_addr = addr
                    break
            self.poll_addr = new_poll_addr
            if self.poll_addr > 0:
                self.get_status(self.poll_addr)
            # Rischedula la richiesta
            self.poll_event = Clock.schedule_once(self.polling_callback, 1)

    def send_status_callback(self, dt):
        mqtt = mqtt_manager()
        mqtt.send_status()
