import paho.mqtt.client as mqtt
import ssl
import time
import os
import logging
import sys
import configparser
import platform
import threading
from flufl.i18n import initialize
from mqtt_manager import mqtt_manager
from device_manager import device_manager
from random import random
from kivy.app import App
from kivy.uix.image import Image
from kivy.uix.widget import Widget
from kivy.uix.button import Button
from kivy.uix.togglebutton import ToggleButton
from kivy.graphics import Color, Ellipse, Line, Canvas, Translate, Fbo, ClearColor, ClearBuffers, Scale
from kivy.lang import Builder
from kivy.resources import resource_add_path
from kivy.uix.screenmanager import ScreenManager, Screen, FadeTransition, NoTransition
from kivy.gesture import Gesture, GestureDatabase
from kivy.clock import Clock, mainthread
from kivy.config import Config
from kivy.properties import ObjectProperty, StringProperty, NumericProperty, ListProperty
from flat_kivy.flatapp import FlatApp
from flat_kivy.uix.flattextinput import FlatTextInput
from functools import partial
from subprocess import call
from __version__ import __version__
if platform.machine().startswith('arm'):
    import picamera
    import picamera.array
    import PIL
    import zbar
    import cv2

_ = initialize('flufl')

class BaseScreen(Screen):
    list_of_prev_screens = []
    gestures = GestureDatabase()
    has_square_gesture = False
    touch_screen_label = StringProperty(_("Tocca lo schermo per iniziare"))

    def __init__(self, name):
        super(BaseScreen, self).__init__(name=name)
        self.logger = logging.getLogger('root')
        self.config_gesture_detected = False
        if not BaseScreen.has_square_gesture:
            gesture = BaseScreen.gestures.str_to_gesture('eNq1mEluIzcYRvd1EXsT4Z+HC6i3AXyAwG0LttEdW7DUSfr2YZHVEgk4KQGCtZH8RD4OH4eSb'
                '1++vfz1c/O0Oxx/vO+mL8v7Hqbbxz1Odzev93/ubqY9lY/ljafD3c3h+P72bXcof8p0+32v0+'
                '2HkrtabNrbrPJSf//28nqcq8VcLf+j2u9zqWmPrQdzF36WKkjTFjasKZKMaJwqGTp355/5a56'
                '2v8FGREAVLIINymebDl/v/78Zqc3o9LS04EgogK6ECeSlhadFTomIHElsKqoS6/I6dPQL5JiW'
                'aSTKYMZEuS6PKs8L5DD03DB11U51+gnX7EVuGOQUgqDIxHGBnKqc1+XkJXLMyAyXUFkPlGqgp'
                'J/irnmSr7s5kThBIhAB5RJ3jZPOcYpbiIG6elkc81Je3DL222TVzTVMPoVJggFESoEGBIJndy'
                'irWWpYRonyAnfNkk9ZQkQYghuLSlnI7tfIa5i8hDmvZFBSUwdH5ixDgKv6XvNkP+mxbDxgo1S'
                '2JIhr3DVPzpOb0IGQ3DTVhTPP8nEH+bpcaqCCJzkHOjlqpISFAp/lZMqMZqIBLnCBvCYqfJKL'
                'EqNyCgkFFNlZjt0qJaD1rS81UTknKuEGqYgRGUZdnsxlnapilrkiKafjurzmKec8Sy12KTPvT'
                'mWn5lle9mS/B2z9MJcaqOSqvC7TflRlkZZ7bM2vNVPFT/PXWJUv8ZO5hkiAQVkz5Wpat9dcVT'
                '/JXoNVv8Quoz3Wl43WZDU/x241V1vfq+UrRTFhLicyAxpecG1YTdXOqVLXP4XSRmcHCg/NIPG'
                'AcnOs22uqdk4Vx777dfaaqp2uU+jPWGdkvcpeU7X8HLvXVP10pWJKOdSFksIMSgNXyWuofrpT'
                'yyUKaewgxuCJeJW8ZuqnB6TxIATspkXGJ16r8vkHwMP7bvd6epx3m5/ny0V/uy1X4wamLaWVt'
                '+PeY7qfoQ8wG8wGvcKACjMGiDMsj/EDpApRB8gV0lhdKpSxulaoI7QKDQfoDfIAY4Cyyf4lc4'
                'lsJaSV4FotYYDzCd69bC6BfYloU5XUQ88GeYBLA9JD+6BfqX0J1VZtGThVKD72i+cS3k+NUKv'
                'WZkFbhMwfVMu+BOXYHy8lENqcaMuM2vyWp+WB2kJpoLrQNhOqI21Toa3HJAttw5c2fOKFtvGz'
                'jrSNmWikbdAAI11WMw8U23L2HCkOdBkxUk8/msyyCVsj0IeAKAP9aI5R+yJ1+mZqPXVYqPc0f'
                'jUSA82FZk9zmWNqo45hG5WfSB1lWOaCqDMwLDkTD/SXoQy0nTzPu5en52P9f4BO25jbLfTvl8'
                'fjc4VWoDZ4fPu+e79/fdjVL7we0jC+cC63nJJ/7N/fHn88NHdM29xw+VUXOp+Mzm7FcPi6+Re'
                'GcFi7')
            gesture.name = "square"
            BaseScreen.gestures.add_gesture(gesture)
            BaseScreen.has_square_gesture = True
        self.enabled_go_to_config = True

    @mainthread
    def go_to_previous_screen(self):
        #self.logger.debug('go_to_previous_screen ' + str(BaseScreen.list_of_prev_screens) + " " + str(threading.current_thread()))
        if len(BaseScreen.list_of_prev_screens) >= 2:
            BaseScreen.list_of_prev_screens.pop() # Remove current from list
            self.manager.current = BaseScreen.list_of_prev_screens.pop() # Go to previous

    def on_enter(self):
        super(BaseScreen, self).on_enter()
        BaseScreen.list_of_prev_screens.append(self.name)
        # Max history 20 screens
        BaseScreen.list_of_prev_screens = BaseScreen.list_of_prev_screens[-20:]
        self.logger.debug('Ingresso schermata ' + self.__class__.__name__)

    def on_leave(self):
        super(BaseScreen, self).on_leave()
        self.logger.debug('Uscita schermata ' + self.__class__.__name__)

    def on_touch_down(self, touch):
        super(BaseScreen, self).on_touch_down(touch)
        if self.enabled_go_to_config:
            try:
                touch.ud['gesture_line'] = Line(points=(touch.x, touch.y))
            except Exception as e:
                pass
 
    def on_touch_move(self, touch):
        super(BaseScreen, self).on_touch_move(touch)
        if self.enabled_go_to_config:
            try:
                touch.ud['gesture_line'].points += [touch.x, touch.y]
            except Exception as e:
                pass
 
    def on_touch_up(self, touch):
        super(BaseScreen, self).on_touch_up(touch)
        self.config_gesture_detected = False
        if self.enabled_go_to_config:
            try:
                gesture = Gesture()
                gesture.add_stroke(
                    zip(touch.ud['gesture_line'].points[::2],
                        touch.ud['gesture_line'].points[1::2]))
                gesture.normalize()
                match = BaseScreen.gestures.find(gesture, minscore=0.70)
                if match:
                    if match[1].name == "square":
                        self.config_gesture_detected = True
                        SostenibileBoxApp().raise_numpad(_("Inserisci la password:"), self.on_enter_admin_password, timeout=300)
            except Exception as e:
                pass

    def on_enter_admin_password(self, value, is_return): 
        if is_return and (value == 942850 or value == 0): # Temporaneo!!!!
            self.manager.current = 'config'
 
class SplashScreen(BaseScreen):
    def on_touch_up(self, touch):
        super(SplashScreen, self).on_touch_up(touch)
        if not self.config_gesture_detected:
            self.manager.current = 'choose_code'

class SetTempWidget(Widget):
    set_temp_label = StringProperty(_("Imposta temp\ntutti box"))

    def __init__(self, **kwargs):
        super(SetTempWidget, self).__init__(**kwargs)

    def on_set_temp(self):
        device = device_manager()
        boxes_db = device.get_boxes_db()
        for addr in boxes_db.keys():
            device.get_status(addr)

class ChooseCodeScreen(BaseScreen):
    xhint = NumericProperty(.3)
    yhint = NumericProperty(.3)
    title_text = ObjectProperty(None)
    enter_qrcode_label = StringProperty(_("Mostra\ncodice alla\ntelecamera"))
    enter_manual_code_label = StringProperty(_("Digita\ncodice"))
    title_label = StringProperty(_("Ritiro prodotto"))

    def __init__(self, name):
        super(ChooseCodeScreen, self).__init__(name=name)
        self.enter_qrcode_button.font_size = 22
        self.enter_manual_code_button.font_size = 22

    def on_enter(self):
        super(ChooseCodeScreen, self).on_enter()
        self.title_text.text = self.title_label

    def on_qrcode(self):
        pass

    def on_manual_code(self):
        pass

class ConfigScreen(BaseScreen):
    xhint = NumericProperty(.2)
    yhint = NumericProperty(.2)
    status_text = ObjectProperty(None)
    hardware_detection_label = StringProperty(_("Riconoscimento\nhardware"))
    open_doors_label = StringProperty(_("Apertura\ntutte\nporte"))
    enable_all_boxes_label = StringProperty(_("Abilita\ntutti\nbox"))
    disable_all_boxes_label = StringProperty(_("Disabilita\ntutti\nbox"))
    camera_label = StringProperty(_("Camera"))
    exit_label = StringProperty(_("Uscita\nprogramma"))

    def __init__(self, name):
        super(ConfigScreen, self).__init__(name=name)
        self.enabled_go_to_config = False
        self.requested_camera = False

    def on_enter(self):
        super(ConfigScreen, self).on_enter()
        if self.requested_camera:
            self.requested_camera = False
            self.status_text.text = str(CameraScreen.qrcode)

    def on_detect_boxes(self):
        device = device_manager()
        device.detect_boxes()
        boxes_db = device.get_boxes_db()
        device.log_available_boxes(boxes_db)

    def on_open_doors(self):
        device = device_manager()
        boxes_db = device.get_boxes_db()
        for addr in boxes_db.keys():
            device.open_door(addr, True)

    def on_enable_all_boxes(self):
        device = device_manager()
        boxes_db = device.get_boxes_db()
        for addr in boxes_db.keys():
            device.enable(addr, True)

    def on_disable_all_boxes(self):
        device = device_manager()
        boxes_db = device.get_boxes_db()
        for addr in boxes_db.keys():
            device.enable(addr, False)

    def on_get_status(self):
        device = device_manager()
        boxes_db = device.get_boxes_db()
        for addr in boxes_db.keys():
            device.get_status(addr)

    def on_camera(self):
        self.requested_camera = True
        self.manager.current = 'camera'

    def on_exit_to_shell(self):
        if platform.machine().startswith('arm'):
            call(["killall", "supervisord"])
        sys.exit(0)

class CameraScreen(BaseScreen):
    xhint = NumericProperty(.2)
    yhint = NumericProperty(.2)
    show_qrcode_label = StringProperty(_("Mostra il codice alla telecamera"))
    status_text = ObjectProperty(None)
    camera = None
    overlay = None
    scanner = None
    rawCapture = None
    stream = None
    thread = None
    stopped = False
    timer_start = None
    qrcode = None

    def __init__(self, name):
        super(CameraScreen, self).__init__(name=name)
        self.enabled_go_to_config = False

    def export_to_png(self, filename, *args):
        '''Saves an image of the widget and its children in png format at the
        specified filename. Works by removing the widget canvas from its
        parent, rendering to an :class:`~kivy.graphics.fbo.Fbo`, and calling
        :meth:`~kivy.graphics.texture.Texture.save`.
        .. note::
            The image includes only this widget and its children. If you want
            to include widgets elsewhere in the tree, you must call
            :meth:`~Widget.export_to_png` from their common parent, or use
            :meth:`~kivy.core.window.WindowBase.screenshot` to capture the
            whole window.
        .. note::
            The image will be saved in png format, you should include the
            extension in your filename.
        .. versionadded:: 1.9.0
        '''

        if self.parent is not None:
            canvas_parent_index = self.parent.canvas.indexof(self.canvas)
            if canvas_parent_index > -1:
                self.parent.canvas.remove(self.canvas)

        fbo = Fbo(size=self.size, with_stencilbuffer=True)

        with fbo:
            ClearColor(0, 0, 0, 0)
            ClearBuffers()
            Scale(1, -1, 1)
            Translate(-self.x, -self.y - self.height, 0)

        fbo.add(self.canvas)
        fbo.draw()
        fbo.texture.save(filename, flipped=False)
        fbo.remove(self.canvas)

        if self.parent is not None and canvas_parent_index > -1:
            self.parent.canvas.insert(canvas_parent_index, self.canvas)

        return True

    def on_enter(self):
        super(CameraScreen, self).on_enter()
        self.status_text.text = self.show_qrcode_label
        CameraScreen.timer_start = Clock.schedule_once(self.start_camera, .1) # Necessario avviare la camera ritardata altrimenti si corrompe l'icona dell'indietro

    def start_camera(self, dt):
        CameraScreen.timer_start = None
        if CameraScreen.camera is None:
            CameraScreen.camera = picamera.PiCamera()
            CameraScreen.camera.framerate = 10
        if CameraScreen.scanner is None:
            CameraScreen.scanner = zbar.ImageScanner()
            CameraScreen.scanner.parse_config('enable')
        if CameraScreen.rawCapture is None:
            CameraScreen.rawCapture = picamera.array.PiRGBArray(CameraScreen.camera, size=CameraScreen.camera.resolution)
        if CameraScreen.stream is None:
            CameraScreen.stream = CameraScreen.camera.capture_continuous(CameraScreen.rawCapture, format="bgr", use_video_port=True)
        CameraScreen.camera.start_preview()
        self.export_to_png('/tmp/camera.png')
        img = PIL.Image.open('/tmp/camera.png')
        CameraScreen.overlay = CameraScreen.camera.add_overlay(img.tobytes(), size=img.size)
        CameraScreen.overlay.alpha = 128
        CameraScreen.overlay.layer = 3
        CameraScreen.stopped = False
        CameraScreen.qrcode = None
        if CameraScreen.thread is None:
            CameraScreen.thread = threading.Thread(target=self.update_camera, args=())
            CameraScreen.thread.start()

    def update_camera(self):
        # keep looping infinitely until the thread is stopped
        self.logger.debug("Avvio thread camera")
        try:
            for frame in CameraScreen.stream:
                # grab the frame from the CameraScreen.stream and clear the CameraScreen.stream in
                # preparation for the next frame
                output = frame.array
                gray = cv2.cvtColor(output, cv2.COLOR_BGR2GRAY, dstCn=0)
                pil = PIL.Image.fromarray(gray)
                width, height = pil.size
                raw = pil.tobytes()
                image = zbar.Image(width, height, 'Y800', raw)
                CameraScreen.scanner.scan(image)
                for symbol in image:
                    CameraScreen.qrcode = symbol.data
                    self.logger.debug('Decoded {} symbol: "{}"'.format(symbol.type, symbol.data))
                    break
                CameraScreen.rawCapture.truncate(0)
                if CameraScreen.stopped or CameraScreen.qrcode is not None:
                    break
        finally:
            self.logger.debug("Arresto thread camera")
            if not CameraScreen.stopped and CameraScreen.qrcode is not None:
                self.go_to_previous_screen()

    def on_leave(self):
        super(CameraScreen, self).on_leave()
        self.logger.debug('CameraScreen on_leave')
        CameraScreen.stopped = True
        if CameraScreen.timer_start is not None:
            CameraScreen.timer_start.cancel()
            CameraScreen.timer_start = None
        if CameraScreen.thread is not None:
            try:
                CameraScreen.thread.join(10)
            except:
                self.logger.error("Errore join() sul thread della camera")
            CameraScreen.thread = None
        if CameraScreen.stream is not None:
            CameraScreen.stream.close()
            CameraScreen.stream = None
        if CameraScreen.rawCapture is not None:
            CameraScreen.rawCapture.close()
            CameraScreen.rawCapture = None
        if CameraScreen.camera is not None:
            CameraScreen.camera.stop_preview()
            CameraScreen.camera.remove_overlay(CameraScreen.overlay)
            CameraScreen.camera.close()
            CameraScreen.camera = None

class SostenibileBoxApp(FlatApp):

    def build(self):
        sm = ScreenManager(transition=NoTransition())
        sm.add_widget(SplashScreen(name='splash'))
        sm.add_widget(ConfigScreen(name='config'))
        sm.add_widget(CameraScreen(name='camera'))
        sm.add_widget(ChooseCodeScreen(name='choose_code'))
        return sm

    def setup_themes(self):
        main = {
            'FlatButton': {
                'color_tuple': ('Gray', '0000'),
                'font_color_tuple': ('LightGreen', '800'),
                'style': 'Button',
                },
            'RaisedFlatButton': {
                'color_tuple': ('Gray', '0000'),
                'font_color_tuple': ('LightGreen', '800'),
                'style': 'Button',
                },
            'FlatLabel': {
                'style': 'Button',
                },
            'FlatSlider': {
                'bar_fill_color_tuple': ('LightGreen', '500'),
                'handle_accent_color_tuple': ('LightGreen', '200'),
                }
            }

        accent = {
            'FlatButton': {
                'color_tuple': ('LightGreen', '500'),
                'font_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                },
            'RaisedFlatButton': {
                'color_tuple': ('LightGreen', '500'),
                'font_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                },
            'FlatIconButton': {
                'color_tuple': ('LightGreen', '500'),
                'font_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                'icon_color_tuple': ('Gray', '1000')
                },
            'FlatToggleButton': {
                'color_tuple': ('LightGreen', '500'),
                'font_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                },
            'RaisedFlatToggleButton': {
                'color_tuple': ('LightGreen', '500'),
                'font_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                },
            'FlatCheckBox': {
                'color_tuple': ('Gray', '0000'),
                'check_color_tuple': ('LightGreen', '500'),
                'outline_color_tuple': ('Gray', '1000'),
                'style': 'Button',
                'check_scale': .7,
                'outline_size': '10dp',
                },
            'FlatCheckBoxListItem': {
                'font_color_tuple': ('Gray', '1000'),
                'check_color_tuple': ('LightGreen', '500'),
                'outline_color_tuple': ('Gray', '800'),
                'style': 'Button',
                'check_scale': .7,
                'outline_size': '10dp',
                },
            }

        self.theme_manager.add_theme('green', 'main', main)
        self.theme_manager.add_theme('green', 'accent', accent)
